//+------------------------------------------------------------------+
//|                                                      CashbackViewer.mq4 |
//|                             Copyright © 2025 FX Insight 360      |
//|                                       https://fxinsight360.com/        |
//+------------------------------------------------------------------+
#property copyright "Copyright © 2025 FX Insight 360"
#property link      "https://fxinsight360.com/"
#property version   "1.00"
#property strict
#property indicator_chart_window



// インジケーターのプロパティ
#property copyright "https://fxinsight360.com/"
#property link      "https://fxinsight360.com/"
#property version   "1.0"
#property indicator_chart_window

// パラメーター設定（固定スプレッド、JPY基準）
input double Spread_USDJPYm = 1.0;  // USDJPY
input double Spread_EURJPYm = 2.0;  // EURJPY
input double Spread_GBPJPYm = 2.2;  // GBPJPY
input double Spread_EURUSDm = 1.0;  // EURUSD
input double Spread_GBPUSDm = 1.0;  // GBPUSD
input double Spread_AUDJPYm = 1.8;  // AUDJPY
input double Spread_AUDUSDm = 1.0;  // AUDUSD
input double Spread_EURAUDm = 2.0;  // EURAUD
input double Spread_GBPAUDm = 2.5;  // GBPAUD
input double Spread_EURGBPm = 2.0;  // EURGBP

input double CashbackRate = 0.4;     // キャッシュバック率
input int X_Position = 10;          // 表示位置のX座標
input int Y_Position = 10;          // 表示位置のY座標
input color TextColor = clrBlack;   // テキストカラー
input int FontSize = 12;            // フォントサイズ

double TotalCashback = 0;
double TotalLots = 0;

// 初期化関数
int OnInit()
{
    // ヘッダーラベルを作成
    CreateLabel("CashbackHeader", "Symbol | Lots | Cashback (JPY)", X_Position, Y_Position);
    return(INIT_SUCCEEDED);
}

// 終了時にオブジェクトを削除
void OnDeinit(const int reason)
{
    ObjectsDeleteAll(0, "CashbackData_");
    ObjectDelete(0, "CashbackHeader");
}

// インジケーターのメイン計算
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
    // 表示データのリセット
    ObjectsDeleteAll(0, "CashbackData_");
    
    TotalCashback = 0;
    TotalLots = 0;
    
    string symbols[];
    double lots[];
    double cashbacks[];
    
    // 履歴注文を処理
    int totalOrders = OrdersHistoryTotal();
    for(int i = 0; i < totalOrders; i++)
    {
        if(!OrderSelect(i, SELECT_BY_POS, MODE_HISTORY)) continue;
        if(OrderType() > OP_SELL) continue; // 買い・売り注文以外は無視
        
        string symbol = OrderSymbol();
        double lotSize = OrderLots();
        double spread_points = GetFixedSpread(symbol); // 固定スプレッドを取得
        
        // キャッシュバックを計算（JPY基準）
        // スプレッド * キャッシュバック率 * ロット数 * 1000（JPY単位）
        double cashback = spread_points * CashbackRate * lotSize * 1000;

        // 配列に追加または更新
        int index = FindSymbolIndex(symbols, symbol);
        if(index == -1)
        {
            ArrayResize(symbols, ArraySize(symbols) + 1);
            ArrayResize(lots, ArraySize(lots) + 1);
            ArrayResize(cashbacks, ArraySize(cashbacks) + 1);
            index = ArraySize(symbols) - 1;
            symbols[index] = symbol;
            lots[index] = 0;
            cashbacks[index] = 0;
        }
        
        lots[index] += lotSize;
        cashbacks[index] += cashback;
        
        TotalLots += lotSize;
        TotalCashback += cashback;
    }
    
    // 表示を更新
    int yOffset = Y_Position + 20;
    for(int j = 0; j < ArraySize(symbols); j++)
    {
        string labelName = "CashbackData_" + IntegerToString(j);
        string displayText = StringFormat("%s | %.2f | %d JPY", 
            symbols[j], lots[j], (int)cashbacks[j]); // 小数点なしに変換
        CreateLabel(labelName, displayText, X_Position, yOffset);
        yOffset += 20;
    }
    
    // 合計の表示
    CreateLabel("CashbackData_Total", 
        StringFormat("Total   |   %.2f   |   %d JPY",  TotalLots, (int)TotalCashback), // 小数点なし
        X_Position, yOffset);
    
    return(rates_total);
}

// シンボルごとの固定スプレッドを取得
double GetFixedSpread(string symbol)
{
    if(symbol == "USDJPYm") return Spread_USDJPYm;
    if(symbol == "EURJPYm") return Spread_EURJPYm;
    if(symbol == "GBPJPYm") return Spread_GBPJPYm;
    if(symbol == "EURUSDm") return Spread_EURUSDm;
    if(symbol == "GBPUSDm") return Spread_GBPUSDm;
    if(symbol == "AUDJPYm") return Spread_AUDJPYm;
    if(symbol == "AUDUSDm") return Spread_AUDUSDm;
    if(symbol == "EURAUDm") return Spread_EURAUDm;
    if(symbol == "GBPAUDm") return Spread_GBPAUDm; // 修正済み
    return 0; // デフォルト値（見つからない場合）
}

// 配列内のシンボルインデックスを探す
int FindSymbolIndex(string &array[], string value)
{
    for(int i = 0; i < ArraySize(array); i++)
        if(array[i] == value) return i;
    return -1;
}

// ラベルを作成または更新
void CreateLabel(string name, string text, int x, int y)
{
    if(ObjectFind(0, name) == -1)
        ObjectCreate(0, name, OBJ_LABEL, 0, 0, 0);
    
    ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
    ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y);
    ObjectSetInteger(0, name, OBJPROP_CORNER, CORNER_LEFT_UPPER);
    ObjectSetText(name, text, FontSize, "Arial", TextColor);
}

